package election;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * A Riding is an area in a country that elects someone to government.
 * Store the name of the riding, plus track all the candidates by storing
 * them in a map that tracks not only who they are, but also counts their votes.
 * The riding can return who was elected in the riding.
 */
public class Riding {
    private final String name;
    private final Map<Candidate, Integer> votes = new HashMap<>();

    public String getName() {
        return name;
    }

    public Riding(String name) {
        this.name = name;
    }

    public void addCandidate(Candidate candidate) {
        assert !votes.containsKey(candidate);
        votes.put(candidate, 0);
    }

    public Iterable<Candidate> getCandidates() {
        return new Iterable<Candidate>() {
            @Override
            public Iterator<Candidate> iterator() {
                return votes.keySet().iterator();
            }
        };
    }

    public int getVotesForCandidate(Candidate candidate) {
        return votes.get(candidate);
    }

    public Candidate getElectedCandidate() {
        if (votes.isEmpty()) {
            return null;
        }
        boolean tie = false;

        Candidate leading = null;
        for (Candidate c : votes.keySet()) {
            if (leading == null) {
                leading = c;
            } else if (votes.get(c) > votes.get(leading)) {
                leading = c;
                tie = false;
            } else if (votes.get(c) == votes.get(leading)) {
                tie = true;
            }
        }
        if (tie) {
            return null;
        } else {
            return leading;
        }
    }

    public void voteFor(Candidate candidate) {
        int current = votes.getOrDefault(candidate, 0);
        votes.put(candidate, current + 1);
    }
}
